#include <string>
#include <stdio.h>
#include <sstream>
#include "util.h"

string checkmac(const char* mac) {
	assert(mac);

	string m = mac;

	if (m.npos != m.find_first_not_of("0123456789ABCDEFabcdef:")) 
		goto return_empty;

	if (m.size() != 6*3-1)
		goto return_empty;

	for(int i=0; i<6*3-1; i++) { // 6 parts
		if (i%3 == 2) {
			if (m[i] != ':')
				goto return_empty;
		} else {
			if (!isxdigit(m[i]))
				goto return_empty;
		}
	}
	
	return m;
	
return_empty:
	m = "";
	return m;
}

DebugPrint::DebugPrint(string prefix, level_t level) {
	this->prefix = prefix;

	setLevel(level);
}

void DebugPrint::setLevel(int level) {
	setLevel((level_t) level);
}

void DebugPrint::setLevel(level_t level) {

	if (level < 0)
		level = (level_t) 0;

	this->level = level;
}

void DebugPrint::printError(const char* fmt, ...) {
	va_list ap;

	fputs(prefix.c_str(), stderr);
	fputs("ERROR: ", stderr);

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}	

void DebugPrint::printWarning(const char* fmt, ...) {
	va_list ap;

	if (level < warning)
		return;
	
	fputs(prefix.c_str(), stderr);
	fputs("Warning: ", stderr);

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
}	

void DebugPrint::print(const char* fmt, ...) {
	va_list ap;

	if (level < normal)
		return;
	
	fputs(prefix.c_str(), stdout);

	va_start(ap, fmt);
	vprintf(fmt, ap);
	va_end(ap);
}	

void DebugPrint::printVerbose(const char* fmt, ...) {
	va_list ap;

	if (level < verbose)
		return;
	
	fputs(prefix.c_str(), stdout);

	va_start(ap, fmt);
	vprintf(fmt, ap);
	va_end(ap);
}	

void DebugPrint::printDebug(const char* fmt, ...) {
	va_list ap;

	if (level < debug)
		return;
	
	fputs(prefix.c_str(), stdout);
	fputs("debug: ", stdout);

	va_start(ap, fmt);
	vprintf(fmt, ap);
	va_end(ap);
}	

void DebugPrint::printDebug2(const char* fmt, ...) {
	va_list ap;

	if (level < debug2)
		return;
	
	fputs(prefix.c_str(), stdout);
	fputs("debug2: ", stdout);

	va_start(ap, fmt);
	vprintf(fmt, ap);
	va_end(ap);
}	

string DebugPrint::printVector(char* pre, const vector<byte>& vec, 
		string delimiter) {
	
	ostringstream ss;

	ss << pre;

	vector<byte>::const_iterator i;
	
	for(i=vec.begin(); i!=vec.end();) {
		ss.setf(ios::hex, ios::basefield);
		ss.width(2);
		ss.fill('0');
		ss << (int)*i;
		if (++i!=vec.end())
			ss << delimiter;
	}
	return ss.str();
}

string DebugPrint::printVector(char* pre, byte *vec, int N, string delimiter) {
	
	ostringstream ss;

	ss << pre;

	byte *i;
	
	for(i=vec; i!=vec+N;) {
		ss.setf(ios::hex, ios::basefield);
		ss.width(2);
		ss.fill('0');
		ss << (int)*i;
		if (++i!=vec+N)
			ss << delimiter;
	}
	return ss.str();
}

void CRC32::Init() {
	crc = 0x00000000;
	counter = 0;
}

void CRC32::Update(char data) {
	unsigned char swdata = swap8(data);

	for (int bit=0; bit<8; bit++) {
		if (crc & 0x80000000)  // top bit == 1
			crc = ((crc << 1) | (swdata >> 7)) ^ 0x04c11db7; // shift + XOR poly
		else
			crc = ((crc << 1) | (swdata >> 7)); // shift left

		swdata = swdata << 1;
		//printf("%08x\n", crc);
	}
	
	if (counter < 4)
		crc = crc ^ 0x000000ff;

	counter++;

	//printf("%08x\n\n", crc);
}

void CRC32::Update(char* data, int len) {
	
	for(int i=0; i<len; i++)
		Update(data[i]);
	
}

void CRC32::Finish() {

	for(int bit=0; bit<32; bit++) { 
		if (crc & 0x80000000) 
			crc = (crc << 1) ^ 0x04c11db7;
		else
			crc = (crc << 1);
		//printf("%08x\n", crc);
	}

	crc = crc ^ 0xffffffff;
	//printf("%08x\n", crc);
}

unsigned char CRC32::operator[] (int i) {
	return (getCRC() >> (i*8)) & 0xff;
}

// get bits from the top and push them in the reverse order
unsigned char CRC32::swap8(unsigned char x) {
	unsigned char y=0;
	int i;
	for (i=0; i<8; i++) {
		y = y >> 1;
		if (x & 0x80) 
			y = y | 0x80;

		x = x << 1;
	}
	return y;
}

unsigned int CRC32::swap32(unsigned int x) {
	unsigned int y=0;
	int i;
	for (i=0; i<32; i++) {
		y = y >> 1;
		if (x & 0x80000000) 
			y = y | 0x80000000;

		x = x << 1;
	}
	return y;
}
/*
int main() {
	string mac = "0000000f0000";

	printf("%s -> %s\n", mac.c_str(), checkmac(mac.c_str()).c_str());
	*/
/*
	CRC32 crc;

	crc.Init();
	crc.Update(0);
	crc.Finish();
	printf("%08x\n", crc.getCRC());

	return 0;
}
*/
