#ifndef WEP_CRACK_H
#define WEP_CRACK_H

#include <cassert>
#include <pthread.h>
#include <list>
#include <vector>
#include <map>
#include <algorithm>
#include <numeric>
#include <functional>
#include <unistd.h>
#include <stdio.h>


typedef unsigned char byte;

#include "util.h"

struct WeakIV {
	vector<byte> IV;
	byte Z;		// first output byte of PRGA
};

class CrackThread {
	private:
		const int I;	// IV len
		const int l;	// key len
		const int N;	// n is word len, N=2^n
		const int B;	// key byte index this thread is cracking
		vector<int> stats;	// statistics for this byte
		const vector<byte> key;	// contains key bytes < B
		const list<WeakIV>& IVs;// list of weak IVs
		pthread_t thread;	// crack thread descriptor

		int hits;
		int hitsFMS;
		int processed;
		DebugPrint dp;
		
		void crack();
		int crackByte(byte* K, byte Z, bool& two_equal);

	public:
		CrackThread(int I, int l, int N, int B, const vector<byte>& key,
				const list<WeakIV>& ivs, int debug_level);
		~CrackThread();

		int getB() { return B; }
		int getHits() { return hits; }
		int getHitsFMS() { return hitsFMS; }
		int getProcessed() { return processed; }
		const vector<int>& getStats() { return stats; }
		const vector<byte>& getKey() { return key; }

		//void setDebugLevel(int l) { dp.setLevel(l); }

		friend void* CrackThreadStart(void* obj);
};

class Monitor {
	private:
		int I;
		int l;
		int N;
		list<WeakIV> IVs;

		int topN;
		int depth;
		int forkLimit;
		int topNhi;
		int loadlimit;
		list<vector<byte> > keys;

		CrackThread* root;
		multimap<CrackThread*, CrackThread*> threads;
		pthread_t monitor_thread;
		pthread_mutex_t keys_lock;

		DebugPrint dp;
		int crackDL;

		void monitor();
		void check_thread(CrackThread* thread, int level);
		void print_threads_recursive(CrackThread* thread, int level);
		void erase_thread_recursive(CrackThread*, CrackThread*);
	public:
		Monitor(int I, int l, int N, int topN, int depthM, int forkAt,
				int topNhi, int loadlimit,
			       	int debug_level, int crack_debug_level);
		~Monitor();

		void addIV(WeakIV iv) { IVs.push_back(iv); }
		bool getNextKey(vector<byte>&);

		friend void* MonitorThreadStart(void* obj);
};

#endif // WEP_CRACK_H
