#ifndef GENERATOR_H
#define GENERATOR_H

#include <stdexcept>
#include <vector>
#include <string>
#include <ctype.h>
#include <pcap.h>

#include "WEPCrack.h"
#include "util.h"

class Generator {
	protected:
		DebugPrint dp;

	public:
		virtual bool getWeakIV(WeakIV&) = 0;
		virtual bool getPacket(vector<byte>&) = 0;
		void setDebugLevel(int level) { dp.setLevel(level); }
		bool extractIV(const vector<byte>&, WeakIV&);
};

class BadCaptureSource: public runtime_error { 
	public:
		BadCaptureSource(const string& src): 
		    runtime_error(
			string("Error opening capture source: ") + src) { }
};

class BadMACAddress: public runtime_error { 
	public:
		BadMACAddress(const string& mac): 
			runtime_error(string("Bad MAC address: ") + mac) { }
};

class BadKeyID: public runtime_error { 
	public:
		BadKeyID(char keyid): 
			runtime_error(string("Bad key ID: ") + '0'+keyid) { }
};

class BadDataLinkType: public runtime_error { 
	public:
		BadDataLinkType(const string& type): 
			runtime_error(string("Bad data link type: ") + type) { }
};

class CaptureGenerator: public Generator {
	private:
		pcap_t* handle;

		bool checkMAC(const char* mac);
	public:
		CaptureGenerator(const char* where, const char* mac, int keyid, bool live, int debug_level=2);
		~CaptureGenerator();
		
		bool getWeakIV(WeakIV&);
		bool getPacket(vector<byte>&);
};

class RandomGenerator: public Generator {
	private:
		int I, l, N;
		vector<byte> key;
		short counter;

		byte genByte1(vector<byte> iv);
		void encrypt(vector<byte>& packet); 
	public:
		RandomGenerator(int len, int debug_level=2);
		RandomGenerator(const vector<byte>& key, int debug_level=2);
		
		bool getWeakIV(WeakIV&);
		bool getPacket(vector<byte>&);
};

#endif //CAPTURE_GENERATOR_H
